<?php

namespace App\Livewire\Admin;

use App\Models\Banner;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Layout;

#[Layout('components.admin-layout')]
class Banners extends Component
{
    use WithPagination, WithFileUploads;

    public $search = '';
    public $typeFilter = '';
    public $statusFilter = '';
    public $sortField = 'created_at';
    public $sortDirection = 'desc';
    public $perPage = 10;

    // Form properties
    public $showForm = false;
    public $formType = 'create';
    public $bannerId = null;
    public $title = '';
    public $image;
    public $type = 'home';
    public $sub_title = '';
    public $url = '';
    public $start_date = '';
    public $end_date = '';

    protected $queryString = [
        'search' => ['except' => ''],
        'typeFilter' => ['except' => ''],
        'statusFilter' => ['except' => ''],
        'sortField' => ['except' => 'created_at'],
        'sortDirection' => ['except' => 'desc'],
        'perPage' => ['except' => 10]
    ];

    protected $rules = [
        'title' => 'required|string|max:255',
        'image' => 'required|image|max:2048',
        'type' => 'required|in:home,category,product,promotional,sidebar,footer',
        'sub_title' => 'nullable|string|max:255',
        'url' => 'nullable|url',
        'start_date' => 'nullable|date',
        'end_date' => 'nullable|date|after_or_equal:start_date',
    ];

    public function mount()
    {
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->reset([
            'showForm', 'formType', 'bannerId', 'title', 'image', 'type',
            'sub_title', 'url', 'start_date', 'end_date'
        ]);
        $this->resetErrorBag();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function create()
    {
        $this->resetForm();
        $this->formType = 'create';
        $this->showForm = true;
    }

    public function edit($bannerId)
    {
        $banner = Banner::findOrFail($bannerId);
        
        $this->formType = 'edit';
        $this->bannerId = $banner->id;
        $this->title = $banner->title;
        $this->type = $banner->type;
        $this->sub_title = $banner->sub_title;
        $this->url = $banner->url;
        $this->start_date = $banner->start_date?->format('Y-m-d');
        $this->end_date = $banner->end_date?->format('Y-m-d');
        $this->showForm = true;

        // Update rules for edit
        $this->rules['image'] = 'nullable|image|max:2048';
    }

    public function save()
    {
        $this->validate();

        $data = [
            'title' => $this->title,
            'type' => $this->type,
            'sub_title' => $this->sub_title,
            'url' => $this->url,
            'start_date' => $this->start_date ?: null,
            'end_date' => $this->end_date ?: null,
        ];

        // Handle image upload
        if ($this->image) {
            $imagePath = $this->image->store('banners', 'public');
            $data['image'] = $imagePath;
        }

        if ($this->formType === 'create') {
            Banner::create($data);
            session()->flash('success', 'Banner created successfully.');
        } else {
            $banner = Banner::findOrFail($this->bannerId);
            
            // Delete old image if new one is uploaded
            if ($this->image && $banner->image) {
                Storage::disk('public')->delete($banner->image);
            }
            
            $banner->update($data);
            session()->flash('success', 'Banner updated successfully.');
        }

        $this->resetForm();
    }

    public function delete($bannerId)
    {
        $banner = Banner::findOrFail($bannerId);
        
        // Delete associated image
        if ($banner->image) {
            Storage::disk('public')->delete($banner->image);
        }

        $banner->delete();
        session()->flash('success', 'Banner deleted successfully.');
    }

    public function removeImage()
    {
        if ($this->bannerId) {
            $banner = Banner::findOrFail($this->bannerId);
            if ($banner->image) {
                Storage::disk('public')->delete($banner->image);
                $banner->update(['image' => null]);
                session()->flash('success', 'Banner image removed successfully.');
            }
        }
        $this->image = null;
    }

    public function render()
    {
        $banners = Banner::query()
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('title', 'like', '%' . $this->search . '%')
                      ->orWhere('sub_title', 'like', '%' . $this->search . '%')
                      ->orWhere('url', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->typeFilter, function ($query) {
                $query->where('type', $this->typeFilter);
            })
            ->when($this->statusFilter === 'active', function ($query) {
                $query->active();
            })
            ->when($this->statusFilter === 'inactive', function ($query) {
                $query->where(function ($q) {
                    $q->where('start_date', '>', now()->format('Y-m-d'))
                      ->orWhere('end_date', '<', now()->format('Y-m-d'));
                });
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        return view('livewire.admin.banners', [
            'banners' => $banners
        ]);
    }
}