<?php

namespace App\Livewire\Admin;

use App\Models\Product;
use App\Models\Category;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Str;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Layout;

#[Layout('components.admin-layout')]
class Products extends Component
{
    use WithPagination, WithFileUploads;

    public $search = '';
    public $sortField = 'name';
    public $sortDirection = 'asc';
    public $perPage = 10;

    // Form properties
    public $showForm = false;
    public $formType = 'create'; // 'create' or 'edit'
    public $productId = null;
    public $name = '';
    public $description = '';
    public $price = 0;
    public $stock = 0;
    public $upc = '';
    public $sku = '';
    public $asin = '';
    public $manufacturer = '';
    public $moq = '';
    public $category_id = '';
    public $image;
    public $meta_title = '';
    public $meta_description = '';
    public $meta_keywords = '';
    public $meta_tags = '';
    public $page_schemas = '';
    public $is_active = true;

    // SEO Analysis properties
    public $seoScore = 0;
    public $seoMetrics = [];

    protected $queryString = [
        'search' => ['except' => ''],
        'sortField' => ['except' => 'name'],
        'sortDirection' => ['except' => 'asc'],
        'perPage' => ['except' => 10]
    ];

    protected $rules = [
        'name' => 'required|string|max:255',
        'description' => 'nullable|string',
        'image' => 'nullable|image|max:2048',
        'price' => 'required|numeric|min:0',
        'stock' => 'required|integer|min:0',
        'upc' => 'nullable|string|max:20',
        'sku' => 'nullable|string|max:50',
        'asin' => 'nullable|string|max:20',
        'manufacturer' => 'nullable|string|max:255',
        'moq' => 'nullable|integer|min:1',
        'category_id' => 'nullable|exists:categories,id',
        'image' => 'nullable|image|max:2048',
        'meta_title' => 'nullable|string|max:255',
        'meta_description' => 'nullable|string',
        'meta_keywords' => 'nullable|string',
        'meta_tags' => 'nullable|string',
        'page_schemas' => 'nullable|string',
        'is_active' => 'boolean'
    ];

    public function mount()
    {
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->reset([
            'showForm', 'formType', 'productId', 'name', 'description', 
            'price', 'stock', 'upc', 'sku', 'asin', 'manufacturer', 
            'moq', 'category_id', 'image', 'meta_title', 'meta_description', 
            'meta_keywords', 'meta_tags', 'page_schemas', 'is_active',
            'seoScore', 'seoMetrics'
        ]);
        $this->resetErrorBag();
    }

    public function analyzeSeo()
    {
        $this->seoMetrics = [];
        
        // Analyze Product Name
        $nameLength = strlen($this->name);
        $this->seoMetrics[] = [
            'name' => 'Product Name',
            'status' => $nameLength >= 10 && $nameLength <= 70,
            'message' => "{$nameLength}/70 chars",
            'weight' => 20
        ];

        // Analyze Meta Title
        $metaTitleLength = strlen($this->meta_title);
        $this->seoMetrics[] = [
            'name' => 'Meta Title',
            'status' => $metaTitleLength >= 10 && $metaTitleLength <= 60,
            'message' => "{$metaTitleLength}/60 chars",
            'weight' => 20
        ];

        // Analyze Meta Description
        $metaDescLength = strlen($this->meta_description);
        $this->seoMetrics[] = [
            'name' => 'Meta Description',
            'status' => $metaDescLength >= 120 && $metaDescLength <= 160,
            'message' => "{$metaDescLength}/160 chars",
            'weight' => 15
        ];

        // Analyze Description Content
        $textContent = strip_tags($this->description);
        $wordCount = str_word_count($textContent);
        $this->seoMetrics[] = [
            'name' => 'Description Length',
            'status' => $wordCount >= 150,
            'message' => "{$wordCount}/150+ words",
            'weight' => 15
        ];

        // Analyze Keywords
        $keywordsCount = !empty($this->meta_keywords) ? count(array_filter(array_map('trim', explode(',', $this->meta_keywords)))) : 0;
        $this->seoMetrics[] = [
            'name' => 'Keywords',
            'status' => $keywordsCount >= 3 && $keywordsCount <= 10,
            'message' => "{$keywordsCount} keywords",
            'weight' => 15
        ];

        // Analyze Product Features
        $hasPrice = $this->price > 0;
        $hasStock = $this->stock >= 0;
        $this->seoMetrics[] = [
            'name' => 'Product Details',
            'status' => $hasPrice && $hasStock,
            'message' => $hasPrice && $hasStock ? 'Complete' : 'Incomplete',
            'weight' => 15
        ];

        // Calculate overall score
        $this->calculateSeoScore();
    }

    private function calculateSeoScore()
    {
        $totalWeight = 0;
        $achievedWeight = 0;

        foreach ($this->seoMetrics as $metric) {
            $totalWeight += $metric['weight'];
            if ($metric['status']) {
                $achievedWeight += $metric['weight'];
            }
        }

        $this->seoScore = $totalWeight > 0 ? 
            round(($achievedWeight / $totalWeight) * 100) : 0;
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function create()
    {
        $this->resetForm();
        $this->formType = 'create';
        $this->showForm = true;
        $this->analyzeSeo();
    }

    public function edit($productId)
    {
        $product = Product::findOrFail($productId);
        
        $this->formType = 'edit';
        $this->productId = $product->id;
        $this->name = $product->name;
        $this->description = $product->description;
        $this->price = $product->price;
        $this->stock = $product->stock;
        $this->upc = $product->upc;
        $this->sku = $product->sku;
        $this->asin = $product->asin;
        $this->manufacturer = $product->manufacturer;
        $this->moq = $product->moq;
        $this->category_id = $product->category_id;
        $this->meta_title = $product->meta_title;
        $this->meta_description = $product->meta_description;
        $this->meta_keywords = $product->meta_keywords;
        $this->meta_tags = $product->meta_tags;
        $this->page_schemas = $product->page_schemas;
        $this->is_active = $product->is_active;
        $this->showForm = true;
        $this->analyzeSeo();
    }

    public function save()
    {
        $this->validate();

        $data = [
            'name' => $this->name,
            'slug' => Str::slug($this->name),
            'description' => $this->description,
            'price' => $this->price,
            'stock' => $this->stock,
            'upc' => $this->upc,
            'sku' => $this->sku ?: $this->generateSku(),
            'asin' => $this->asin,
            'manufacturer' => $this->manufacturer,
            'moq' => $this->moq ?: null,
            'category_id' => $this->category_id ?: null,
            'meta_title' => $this->meta_title,
            'meta_description' => $this->meta_description,
            'meta_keywords' => $this->meta_keywords,
            'meta_tags' => $this->meta_tags,
            'page_schemas' => $this->page_schemas,
            'is_active' => $this->is_active,
        ];

        // Handle image upload
        if ($this->image) {
            $imagePath = $this->image->store('products', 'public');
            $data['image'] = $imagePath;
        }

        if ($this->formType === 'create') {
            Product::create($data);
            session()->flash('success', 'Product created successfully.');
        } else {
            $product = Product::findOrFail($this->productId);
            $product->update($data);
            session()->flash('success', 'Product updated successfully.');
        }

        $this->resetForm();
    }

    public function delete($productId)
    {
        $product = Product::findOrFail($productId);
        
        // Delete associated image if exists
        if ($product->image) {
            Storage::disk('public')->delete($product->image);
        }

        $product->delete();
        session()->flash('success', 'Product deleted successfully.');
    }

    public function toggleStatus($productId)
    {
        $product = Product::findOrFail($productId);
        $product->update(['is_active' => !$product->is_active]);
        
        session()->flash('success', 'Product status updated successfully.');
    }

    private function generateSku()
    {
        $prefix = strtoupper(substr($this->name, 0, 3));
        $random = strtoupper(Str::random(6));
        return $prefix . '-' . $random;
    }

    public function render()
    {
        $products = Product::with('category')
            ->when($this->search, function ($query) {
                $query->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%')
                      ->orWhere('sku', 'like', '%' . $this->search . '%')
                      ->orWhere('upc', 'like', '%' . $this->search . '%')
                      ->orWhere('asin', 'like', '%' . $this->search . '%')
                      ->orWhere('manufacturer', 'like', '%' . $this->search . '%');
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        $categories = Category::where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('livewire.admin.products', [
            'products' => $products,
            'categories' => $categories
        ]);
    }
}