<?php

namespace App\Livewire\Admin;

use App\Models\Website;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Log;
use Livewire\Attributes\Layout;
use Livewire\Component;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Exception\ConnectException;

#[Layout('components.admin-layout')]
class SiteDataDashboard extends Component
{
    public $selectedSite = '';
    public $siteData = [];
    public Collection $availableSites;
    public $error = '';

    public function mount()
    {
        $this->availableSites = Website::all();
        
        if ($this->availableSites->count() > 0) {
            $this->selectedSite = $this->availableSites->first()->id;
            $this->loadSiteData();
        }
    }

    // REMOVE THIS METHOD to avoid conflicts
    // public function updatedSelectedSite($value)
    // {
    //     if ($value) {
    //         $this->loadSiteData();
    //     }
    // }

    public function loadSiteData()
    {
        if (!$this->selectedSite) {
            $this->siteData = [];
            $this->dispatch('loadingFinished');
            return;
        }

        try {
            Log::info("Loading site data for site ID: {$this->selectedSite}");
            
            // Dispatch loading started FIRST
            $this->dispatch('loadingStarted');
            
            $this->error = '';
            $this->siteData = [];
            
            $website = Website::find($this->selectedSite);
            
            if ($website) {
                Log::info("Fetching data for website: {$website->name}");
                
                // APIs se real data fetch karo using Guzzle
                $leadsData = $this->fetchLeadsData($website);
                $blogsData = $this->fetchBlogsData($website);
                $categoriesData = $this->fetchCategoriesData($website);
                $productsData = $this->fetchProductsData($website);

                $this->siteData = [
                    // Leads data
                    'total_leads' => $leadsData['total_leads'],
                    'today_leads' => $leadsData['today_leads'],
                    'yesterday_leads' => $leadsData['yesterday_leads'],
                    'current_month_leads' => $leadsData['current_month_leads'],
                    'last_month_leads' => $leadsData['last_month_leads'],
                    'monthly_growth_rate' => $leadsData['monthly_growth_rate'],
                    
                    // Blogs data
                    'total_blogs' => $blogsData['total_blogs'],
                    'published_blogs' => $blogsData['published_blogs'],
                    'draft_blogs' => $blogsData['draft_blogs'],
                    
                    // Categories data
                    'total_categories' => $categoriesData['total_categories'],
                    'active_categories' => $categoriesData['active_categories'],
                    
                    // Products data
                    'total_products' => $productsData['total_products'],
                    'published_products' => $productsData['published_products'],
                    'out_of_stock' => $productsData['out_of_stock'],
                    
                    // Website info
                    'website_name' => $website->name,
                    'website_url' => $website->url,
                    'website_image' => $website->image,
                    
                    // Calculated metrics
                    'conversion_rate' => 0,
                    'growth_rate' => $leadsData['monthly_growth_rate'],
                    
                    'timestamp' => now()->toISOString()
                ];
                
                Log::info("Site data loaded successfully", $this->siteData);
            }
            
        } catch (\Exception $e) {
            Log::error("Error loading site data: " . $e->getMessage());
            $this->error = 'Error loading data: ' . $e->getMessage();
            if (isset($website)) {
                $this->loadDummyData($website);
            }
        } finally {
            // Always dispatch loading finished
            $this->dispatch('loadingFinished');
            Log::info("Loading finished dispatched");
        }
    }

    /**
     * Get Guzzle client instance
     */
    private function getGuzzleClient()
    {
        return new Client([
            'timeout' => 30,
            'connect_timeout' => 10,
            'verify' => false,
            'headers' => [
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                'Accept' => 'application/json',
            ]
        ]);
    }

    /**
     * Fetch leads data using Guzzle
     */
    private function fetchLeadsData($website)
    {
        $client = $this->getGuzzleClient();
        
        try {
            $apiUrl = rtrim($website->url, '/') . '/fetch-leads';
            Log::info("Fetching leads from: {$apiUrl}");

            $response = $client->get($apiUrl);

            if ($response->getStatusCode() === 200) {
                $data = json_decode($response->getBody(), true);
                Log::info("Leads API Response received");
                return $this->calculateLeadMetrics($data);
            } else {
                Log::error("Leads API returned status: " . $response->getStatusCode());
            }

        } catch (ConnectException $e) {
            Log::error("Leads API Connection Error: " . $e->getMessage());
        } catch (RequestException $e) {
            Log::error("Leads API Request Error: " . $e->getMessage());
        } catch (\Exception $e) {
            Log::error("Leads API General Error: " . $e->getMessage());
        }

        return $this->getDefaultLeadMetrics();
    }

    // ... Keep all your other methods EXACTLY as they were
    // fetchBlogsData, fetchCategoriesData, fetchProductsData, 
    // calculateLeadMetrics, getDefaultLeadMetrics, etc.
    // ALL REMAINING METHODS STAY THE SAME

    private function fetchBlogsData($website)
    {
        $client = $this->getGuzzleClient();
        
        try {
            $apiUrl = rtrim($website->url, '/') . '/fetch-blogs';
            Log::info("Fetching blogs from: {$apiUrl}");

            $response = $client->get($apiUrl);

            if ($response->getStatusCode() === 200) {
                $data = json_decode($response->getBody(), true);
                return [
                    'total_blogs' => $data['total_blogs'] ?? $data['count'] ?? 0,
                    'published_blogs' => $data['published_blogs'] ?? $data['published'] ?? 0,
                    'draft_blogs' => $data['draft_blogs'] ?? $data['drafts'] ?? 0,
                ];
            }

        } catch (\Exception $e) {
            Log::error("Blogs API Error: " . $e->getMessage());
        }

        return $this->getDefaultBlogMetrics();
    }

    private function fetchCategoriesData($website)
    {
        $client = $this->getGuzzleClient();
        
        try {
            $apiUrl = rtrim($website->url, '/') . '/fetch-categories';
            Log::info("Fetching categories from: {$apiUrl}");

            $response = $client->get($apiUrl);

            if ($response->getStatusCode() === 200) {
                $data = json_decode($response->getBody(), true);
                return [
                    'total_categories' => $data['total_categories'] ?? $data['count'] ?? 0,
                    'active_categories' => $data['active_categories'] ?? $data['active'] ?? 0,
                ];
            }

        } catch (\Exception $e) {
            Log::error("Categories API Error: " . $e->getMessage());
        }

        return $this->getDefaultCategoryMetrics();
    }

    private function fetchProductsData($website)
    {
        $client = $this->getGuzzleClient();
        
        try {
            $apiUrl = rtrim($website->url, '/') . '/fetch-products';
            Log::info("Fetching products from: {$apiUrl}");

            $response = $client->get($apiUrl);

            if ($response->getStatusCode() === 200) {
                $data = json_decode($response->getBody(), true);
                return [
                    'total_products' => $data['total_products'] ?? $data['count'] ?? 0,
                    'published_products' => $data['published_products'] ?? $data['published'] ?? 0,
                    'out_of_stock' => $data['out_of_stock'] ?? $data['out_of_stock_count'] ?? 0,
                ];
            }

        } catch (\Exception $e) {
            Log::error("Products API Error: " . $e->getMessage());
        }

        return $this->getDefaultProductMetrics();
    }

    private function calculateLeadMetrics($data)
    {
        // ... keep your existing calculateLeadMetrics method exactly as is
        $leads = [];
        
        if (isset($data['leads']) && is_array($data['leads'])) {
            $leads = $data['leads'];
        } elseif (isset($data['data']) && is_array($data['data'])) {
            $leads = $data['data'];
        } elseif (is_array($data) && isset($data[0]) && is_array($data[0])) {
            $leads = $data;
        } elseif (isset($data['total_leads'])) {
            return [
                'total_leads' => $data['total_leads'] ?? 0,
                'today_leads' => $data['today_leads'] ?? $data['new_leads'] ?? 0,
                'yesterday_leads' => $data['yesterday_leads'] ?? 0,
                'current_month_leads' => $data['current_month_leads'] ?? 0,
                'last_month_leads' => $data['last_month_leads'] ?? 0,
                'monthly_growth_rate' => $data['monthly_growth_rate'] ?? 0,
            ];
        }

        $totalLeads = count($leads);
        
        $today = now()->format('Y-m-d');
        $yesterday = now()->subDay()->format('Y-m-d');
        $currentMonthStart = now()->startOfMonth()->format('Y-m-d');
        $currentMonthEnd = now()->endOfMonth()->format('Y-m-d');
        $lastMonthStart = now()->subMonth()->startOfMonth()->format('Y-m-d');
        $lastMonthEnd = now()->subMonth()->endOfMonth()->format('Y-m-d');
        
        $todayLeads = 0;
        $yesterdayLeads = 0;
        $currentMonthLeads = 0;
        $lastMonthLeads = 0;

        foreach ($leads as $lead) {
            $createdAt = $lead['created_at'] ?? $lead['created_date'] ?? $lead['date_created'] ?? $lead['timestamp'] ?? null;
            
            if ($createdAt) {
                try {
                    $leadDate = date('Y-m-d', strtotime($createdAt));
                    
                    if ($leadDate === $today) $todayLeads++;
                    if ($leadDate === $yesterday) $yesterdayLeads++;
                    if ($leadDate >= $currentMonthStart && $leadDate <= $currentMonthEnd) $currentMonthLeads++;
                    if ($leadDate >= $lastMonthStart && $leadDate <= $lastMonthEnd) $lastMonthLeads++;
                } catch (\Exception $e) {
                    continue;
                }
            }
        }

        $monthlyGrowthRate = $this->calculateMonthlyGrowthRate($currentMonthLeads, $lastMonthLeads);

        return [
            'total_leads' => $totalLeads,
            'today_leads' => $todayLeads,
            'yesterday_leads' => $yesterdayLeads,
            'current_month_leads' => $currentMonthLeads,
            'last_month_leads' => $lastMonthLeads,
            'monthly_growth_rate' => $monthlyGrowthRate,
        ];
    }

    private function calculateMonthlyGrowthRate($currentMonth, $lastMonth)
    {
        if ($lastMonth > 0) {
            return (($currentMonth - $lastMonth) / $lastMonth) * 100;
        } elseif ($currentMonth > 0) {
            return 100;
        }
        return 0;
    }

    private function getDefaultLeadMetrics()
    {
        return [
            'total_leads' => 0, 
            'today_leads' => 0, 
            'yesterday_leads' => 0, 
            'current_month_leads' => 0,
            'last_month_leads' => 0,
            'monthly_growth_rate' => 0,
        ];
    }

    private function getDefaultBlogMetrics()
    {
        return ['total_blogs' => 0, 'published_blogs' => 0, 'draft_blogs' => 0];
    }

    private function getDefaultCategoryMetrics()
    {
        return ['total_categories' => 0, 'active_categories' => 0];
    }

    private function getDefaultProductMetrics()
    {
        return ['total_products' => 0, 'published_products' => 0, 'out_of_stock' => 0];
    }

    private function loadDummyData($website)
    {
        if ($website) {
            $this->siteData = [
                'total_leads' => rand(50, 500),
                'today_leads' => rand(1, 20),
                'yesterday_leads' => rand(1, 15),
                'current_month_leads' => rand(30, 100),
                'last_month_leads' => rand(25, 90),
                'monthly_growth_rate' => rand(-10, 30),
                'total_blogs' => rand(5, 50),
                'published_blogs' => rand(5, 45),
                'draft_blogs' => rand(0, 10),
                'total_categories' => rand(3, 20),
                'active_categories' => rand(3, 18),
                'total_products' => rand(10, 200),
                'published_products' => rand(8, 180),
                'out_of_stock' => rand(0, 15),
                'website_name' => $website->name,
                'website_url' => $website->url,
                'website_image' => $website->image,
                'conversion_rate' => 0,
                'growth_rate' => rand(-10, 30),
                'timestamp' => now()->toISOString()
            ];
        }
    }

    public function render()
    {
        return view('livewire.admin.site-data-dashboard');
    }
}