<?php

namespace App\Livewire\Admin;

use App\Models\User;
use App\Models\Role;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Livewire\WithFileUploads;
use Livewire\Attributes\Layout;
use Spatie\Permission\Models\Role as ModelsRole;

#[Layout('components.admin-layout')]
class Users extends Component
{
    use WithPagination, WithFileUploads;

    public $search = '';
    public $roleFilter = '';
    public $perPage = 10;
    public $sortField = 'name';
    public $sortDirection = 'asc';

    // Form properties
    public $showForm = false;
    public $formType = 'create';
    public $userId = null;
    public $name = '';
    public $email = '';
    public $password = '';
    public $password_confirmation = '';
    public $role_id = '';
    public $is_active = true;
    public $phone = '';
    public $address = '';
    public $avatar;

    // Use dynamic validation rules
    protected function rules()
    {
        $rules = [
            'name' => 'required|string|max:255',
            'role_id' => 'required|exists:roles,id',
            'is_active' => 'boolean',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'avatar' => 'nullable|image|max:2048',
        ];

        // Add email validation based on form type
        if ($this->formType === 'create') {
            $rules['email'] = 'required|string|email|max:255|unique:users,email';
            $rules['password'] = 'required|min:8|confirmed';
        } else {
            $rules['email'] = 'required|string|email|max:255|unique:users,email,' . $this->userId;
            $rules['password'] = 'nullable|min:8|confirmed';
        }

        return $rules;
    }

    public function mount()
    {
        // Simple auth check - remove permission checks for now
        // if (!Auth::check()) {
        //     return redirect()->route('login');
        // }
    }

    public function resetForm()
    {
        $this->reset([
            'showForm', 'formType', 'userId', 'name', 'email', 'password',
            'password_confirmation', 'role_id', 'is_active', 'phone', 'address', 'avatar'
        ]);
        $this->resetErrorBag();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function create()
    {
        $this->resetForm();
        $this->formType = 'create';
        $this->showForm = true;
    }

    public function edit($userId)
    {
        $user = User::findOrFail($userId);
        
        $this->formType = 'edit';
        $this->userId = $user->id;
        $this->name = $user->name;
        $this->email = $user->email;
        $this->role_id = $user->role_id;
        $this->is_active = $user->is_active;
        $this->phone = $user->phone;
        $this->address = $user->address;
        $this->showForm = true;
    }

    public function save()
    {
        $this->validate();

        $data = [
            'name' => $this->name,
            'email' => $this->email,
            'role_id' => $this->role_id,
            'is_active' => $this->is_active,
            'phone' => $this->phone,
            'address' => $this->address,
        ];

        if ($this->password) {
            $data['password'] = Hash::make($this->password);
        }

        // Handle avatar upload
        if ($this->avatar) {
            $avatarPath = $this->avatar->store('avatars', 'public');
            $data['avatar'] = $avatarPath;
        }

        if ($this->formType === 'create') {
            User::create($data);
            session()->flash('success', 'User created successfully.');
        } else {
            $user = User::findOrFail($this->userId);
            $user->update($data);
            session()->flash('success', 'User updated successfully.');
        }

        $this->resetForm();
    }

    public function delete($userId)
    {
        $user = User::findOrFail($userId);
        
        // Prevent users from deleting themselves
        if ($user->id === Auth::id()) {
            session()->flash('error', 'You cannot delete your own account.');
            return;
        }

        $user->delete();
        session()->flash('success', 'User deleted successfully.');
    }

    public function toggleStatus($userId)
    {
        $user = User::findOrFail($userId);
        
        // Prevent users from deactivating themselves
        if ($user->id === Auth::id()) {
            session()->flash('error', 'You cannot deactivate your own account.');
            return;
        }

        $user->update(['is_active' => !$user->is_active]);
        session()->flash('success', 'User status updated successfully.');
    }

    public function render()
    {
        $users = User::with('role')
            ->when($this->search, function ($query) {
                $query->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('email', 'like', '%' . $this->search . '%');
            })
            ->when($this->roleFilter, function ($query) {
                $query->where('role_id', $this->roleFilter);
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        $roles = ModelsRole::orderBy('name')->get();

        return view('livewire.admin.users', [
            'users' => $users,
            'roles' => $roles
        ]);
    }
}