<?php

namespace App\Livewire\Admin;

use App\Models\Website;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Layout;
use Livewire\WithFileUploads;

#[Layout('components.admin-layout')]

class Websites extends Component
{
    use WithPagination, WithFileUploads;

    public $name, $url, $image;
    public $editId = null;
    public $showModal = false;
    public $deleteId = null;
    public $showDeleteModal = false;
    public $search = '';

    protected $rules = [
        'name' => 'required|string|max:255',
        'url' => 'required|url|max:255',
        'image' => 'nullable|image|max:1024',
    ];

    public function render()
    {
        $websites = Website::when($this->search, function ($query) {
            return $query->where('name', 'like', '%' . $this->search . '%')
                        ->orWhere('url', 'like', '%' . $this->search . '%');
        })
        ->latest()
        ->paginate(10);

        return view('livewire.admin.websites', compact('websites'));
    }

    public function create()
    {
        $this->resetForm();
        $this->showModal = true;
    }

    public function edit($id)
    {
        $website = Website::findOrFail($id);
        $this->editId = $website->id;
        $this->name = $website->name;
        $this->url = $website->url;
        $this->image = null;
        $this->showModal = true;
    }

    public function save()
    {
        $this->validate();

        $data = [
            'name' => $this->name,
            'url' => $this->url,
        ];

        if ($this->image) {
            $data['image'] = $this->image->store('websites', 'public');
        }

        if ($this->editId) {
            $website = Website::findOrFail($this->editId);
            
            // Delete old image if new image uploaded
            if ($this->image && $website->image) {
                Storage::disk('public')->delete($website->image);
            }
            
            $website->update($data);
            session()->flash('message', 'Website updated successfully!');
        } else {
            Website::create($data);
            session()->flash('message', 'Website created successfully!');
        }

        $this->resetForm();
        $this->showModal = false;
    }

    public function confirmDelete($id)
    {
        $this->deleteId = $id;
        $this->showDeleteModal = true;
    }

    public function delete()
    {
        $website = Website::findOrFail($this->deleteId);
        
        // Delete image if exists
        if ($website->image) {
            Storage::disk('public')->delete($website->image);
        }
        
        $website->delete();
        
        $this->showDeleteModal = false;
        $this->deleteId = null;
        session()->flash('message', 'Website deleted successfully!');
    }

    private function resetForm()
    {
        $this->editId = null;
        $this->name = '';
        $this->url = '';
        $this->image = null;
        $this->resetValidation();
    }
}