<?php

namespace App\Livewire;

use App\Models\Blog;
use Livewire\Component;
use App\Models\Post;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Support\Facades\Response;
use Livewire\Attributes\Layout;

#[Layout('components.admin-layout')]

class SitemapGenerator extends Component
{
    public $sitemapType = 'html'; // 'html' or 'xml'
    public $lastGenerated;
    public $urlCounts = [
        'static' => 0,
        'posts' => 0,
        'categories' => 0,
        'products' => 0,
        'total' => 0
    ];

    public function mount()
    {
        $this->lastGenerated = now()->format('Y-m-d H:i:s');
        $this->updateUrlCounts();
    }

    public function switchSitemap($type)
    {
        $this->sitemapType = $type;
    }

    public function generateSitemap()
    {
        // Simply update the timestamp and counts
        $this->lastGenerated = now()->format('Y-m-d H:i:s');
        $this->updateUrlCounts();
        
        session()->flash('message', 'Sitemap refreshed successfully!');
    }

    private function updateUrlCounts()
    {
        $this->urlCounts = [
            'static' => count($this->getStaticPages()),
            'posts' => Blog::count(),
            'categories' => Category::count(),
            'products' => Product::count(),
        ];

        $this->urlCounts['total'] = array_sum($this->urlCounts);
    }

    public function getStaticPages()
    {
        return [
            '/' => [
                'url' => url('/'),
                'title' => 'Homepage',
                'priority' => '1.0',
                'changefreq' => 'daily',
                'lastmod' => now()->format('Y-m-d')
            ],
            '/about' => [
                'url' => url('/about'),
                'title' => 'About Us',
                'priority' => '0.8',
                'changefreq' => 'monthly',
                'lastmod' => now()->format('Y-m-d')
            ],
            '/contact' => [
                'url' => url('/contact'),
                'title' => 'Contact',
                'priority' => '0.7',
                'changefreq' => 'monthly',
                'lastmod' => now()->format('Y-m-d')
            ],
            '/blog' => [
                'url' => url('/blog'),
                'title' => 'Blog',
                'priority' => '0.9',
                'changefreq' => 'weekly',
                'lastmod' => now()->format('Y-m-d')
            ],
            '/services' => [
                'url' => url('/services'),
                'title' => 'Services',
                'priority' => '0.8',
                'changefreq' => 'monthly',
                'lastmod' => now()->format('Y-m-d')
            ],
        ];
    }

    public function getDynamicContent()
    {
        $content = [];

        // Blog Posts
        $posts = Blog::orderBy('updated_at', 'desc')
            ->get()
            ->map(function ($post) {
                return [
                    'url' => url('/blog/' . $post->slug),
                    'title' => $post->title,
                    'priority' => '0.8',
                    'changefreq' => 'monthly',
                    'lastmod' => $post->updated_at->format('Y-m-d')
                ];
            });

        // Categories
        $categories = Category::get()
            ->map(function ($category) {
                return [
                    'url' => url('/category/' . $category->slug),
                    'title' => $category->name,
                    'priority' => '0.6',
                    'changefreq' => 'weekly',
                    'lastmod' => $category->updated_at->format('Y-m-d')
                ];
            });

        // Products (if you have products)
        $products = Product::get()
            ->map(function ($product) {
                return [
                    'url' => url('/products/' . $product->slug),
                    'title' => $product->name,
                    'priority' => '0.7',
                    'changefreq' => 'weekly',
                    'lastmod' => $product->updated_at->format('Y-m-d')
                ];
            });

        return [
            'posts' => $posts,
            'categories' => $categories,
            'products' => $products
        ];
    }

    public function getAllUrls()
    {
        $staticPages = $this->getStaticPages();
        $dynamicContent = $this->getDynamicContent();

        $allUrls = [];

        // Add static pages
        foreach ($staticPages as $page) {
            $allUrls[] = $page;
        }

        // Add dynamic content
        foreach ($dynamicContent as $type => $items) {
            foreach ($items as $item) {
                $allUrls[] = $item;
            }
        }

        return $allUrls;
    }

    public function downloadXml()
    {
        $urls = $this->getAllUrls();
        
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        
        foreach ($urls as $url) {
            $xml .= "  <url>\n";
            $xml .= "    <loc>" . htmlspecialchars($url['url']) . "</loc>\n";
            $xml .= "    <lastmod>{$url['lastmod']}</lastmod>\n";
            $xml .= "    <changefreq>{$url['changefreq']}</changefreq>\n";
            $xml .= "    <priority>{$url['priority']}</priority>\n";
            $xml .= "  </url>\n";
        }
        
        $xml .= '</urlset>';
        
        return Response::make($xml, 200, [
            'Content-Type' => 'application/xml',
            'Content-Disposition' => 'attachment; filename="sitemap.xml"'
        ]);
    }

    public function viewXml()
    {
        $urls = $this->getAllUrls();
        
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        
        foreach ($urls as $url) {
            $xml .= "  <url>\n";
            $xml .= "    <loc>" . htmlspecialchars($url['url']) . "</loc>\n";
            $xml .= "    <lastmod>{$url['lastmod']}</lastmod>\n";
            $xml .= "    <changefreq>{$url['changefreq']}</changefreq>\n";
            $xml .= "    <priority>{$url['priority']}</priority>\n";
            $xml .= "  </url>\n";
        }
        
        $xml .= '</urlset>';
        
        return Response::make($xml, 200, [
            'Content-Type' => 'application/xml'
        ]);
    }

    public function render()
    {
        $urls = $this->getAllUrls();
        $dynamicContent = $this->getDynamicContent();

        return view('livewire.sitemap-generator', [
            'urls' => $urls,
            'dynamicContent' => $dynamicContent,
            'staticPages' => $this->getStaticPages(),
        ]);
    }
}