<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class Banner extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'image',
        'type',
        'sub_title',
        'url',
        'start_date',
        'end_date',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the image URL.
     */
    public function getImageUrlAttribute()
    {
        return $this->image ? Storage::url($this->image) : null;
    }

    /**
     * Get display type with label.
     */
    public function getDisplayTypeAttribute()
    {
        return ucfirst($this->type);
    }

    /**
     * Get type badge color.
     */
    public function getTypeBadgeAttribute()
    {
        $colors = [
            'home' => 'bg-blue-100 text-blue-800',
            'category' => 'bg-green-100 text-green-800',
            'product' => 'bg-purple-100 text-purple-800',
            'promotional' => 'bg-orange-100 text-orange-800',
            'sidebar' => 'bg-indigo-100 text-indigo-800',
            'footer' => 'bg-gray-100 text-gray-800',
        ];

        return $colors[$this->type] ?? 'bg-gray-100 text-gray-800';
    }

    /**
     * Check if banner is currently active.
     */
    public function getIsActiveAttribute()
    {
        $now = Carbon::now();
        
        if ($this->start_date && $this->end_date) {
            return $now->between($this->start_date, $this->end_date);
        } elseif ($this->start_date) {
            return $now->gte($this->start_date);
        } elseif ($this->end_date) {
            return $now->lte($this->end_date);
        }
        
        return true; // No date restrictions
    }

    /**
     * Get status badge.
     */
    public function getStatusBadgeAttribute()
    {
        return $this->is_active 
            ? 'bg-green-100 text-green-800' 
            : 'bg-red-100 text-red-800';
    }

    /**
     * Get status text.
     */
    public function getStatusTextAttribute()
    {
        return $this->is_active ? 'Active' : 'Inactive';
    }

    /**
     * Scope active banners.
     */
    public function scopeActive($query)
    {
        $now = Carbon::now()->format('Y-m-d');
        
        return $query->where(function ($q) use ($now) {
            $q->whereNull('start_date')
              ->orWhere('start_date', '<=', $now);
        })->where(function ($q) use ($now) {
            $q->whereNull('end_date')
              ->orWhere('end_date', '>=', $now);
        });
    }

    /**
     * Scope by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Scope expiring soon (within 7 days).
     */
    public function scopeExpiringSoon($query)
    {
        $soon = Carbon::now()->addDays(7)->format('Y-m-d');
        
        return $query->whereNotNull('end_date')
                    ->where('end_date', '<=', $soon)
                    ->where('end_date', '>=', Carbon::now()->format('Y-m-d'));
    }

    /**
     * Check if banner has date restrictions.
     */
    public function getHasDateRestrictionsAttribute()
    {
        return $this->start_date || $this->end_date;
    }

    /**
     * Get days remaining (if end date is set).
     */
    public function getDaysRemainingAttribute()
    {
        if (!$this->end_date) {
            return null;
        }

        $now = Carbon::now();
        $end = Carbon::parse($this->end_date);
        
        return $now->diffInDays($end, false); // Negative if expired
    }
}