<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Lead extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'firstname',
        'lastname',
        'companyname',
        'address_primary',
        'address_secondary',
        'country',
        'state',
        'city',
        'email',
        'phone',
        'employeeid',
        'seller_permit',
        'duration',
        'location',
        'period',
        'reference',
        'permit_doc',
        'mail_count',
        'source',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'mail_count' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the full name of the lead.
     */
    public function getFullNameAttribute(): string
    {
        return $this->firstname . ' ' . $this->lastname;
    }

    /**
     * Scope a query to only include leads from a specific source.
     */
    public function scopeFromSource($query, string $source)
    {
        return $query->where('source', $source);
    }

    /**
     * Scope a query to only include leads created today.
     */
    public function scopeToday($query)
    {
        return $query->whereDate('created_at', today());
    }

    /**
     * Scope a query to only include leads created in the last N days.
     */
    public function scopeLastDays($query, int $days)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    /**
     * Scope a query to order by most recent.
     */
    public function scopeRecent($query)
    {
        return $query->orderBy('created_at', 'desc');
    }

    /**
     * Get the formatted phone number.
     */
    public function getFormattedPhoneAttribute(): ?string
    {
        if (!$this->phone) {
            return null;
        }

        // Basic phone formatting - you can customize this
        $phone = preg_replace('/\D/', '', $this->phone);
        
        if (strlen($phone) === 10) {
            return '(' . substr($phone, 0, 3) . ') ' . substr($phone, 3, 3) . '-' . substr($phone, 6, 4);
        }
        
        return $this->phone;
    }

    /**
     * Get the full address.
     */
    public function getFullAddressAttribute(): ?string
    {
        $addressParts = array_filter([
            $this->address_primary,
            $this->address_secondary,
            $this->city,
            $this->state,
            $this->country
        ]);

        return $addressParts ? implode(', ', $addressParts) : null;
    }

    /**
     * Check if lead has permit document.
     */
    public function getHasPermitAttribute(): bool
    {
        return !empty($this->permit_doc);
    }

    /**
     * Increment mail count.
     */
    public function incrementMailCount(): void
    {
        $this->increment('mail_count');
    }

    public function product()
    {
        return $this->belongsTo(Product::class);
    }
}