<?php

namespace App\Services;

use Google\Client;
use Google\Service\AnalyticsData;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class GoogleAnalyticsService
{
    protected $client;
    protected $propertyId;
    protected $isConfigured = false;

    public function __construct()
    {
        $this->client = new Client();
        $this->propertyId = config('services.google.analytics.property_id');
        
        try {
            $this->setupClient();
            $this->isConfigured = true;
        } catch (\Exception $e) {
            Log::error('Google Analytics Service configuration failed: ' . $e->getMessage());
            $this->isConfigured = false;
        }
    }

    protected function setupClient()
    {
        $keyFile = env('GOOGLE_SERVICE_ACCOUNT_KEY_FILE');
        
        if (!file_exists($keyFile)) {
            throw new \Exception("Google service account key file not found at: {$keyFile}");
        }

        $this->client->setAuthConfig($keyFile);
        $this->client->addScope(AnalyticsData::ANALYTICS_READONLY);
    }

    public function isConfigured()
    {
        return $this->isConfigured && !empty($this->propertyId);
    }

    public function getRealTimeData()
    {
        if (!$this->isConfigured()) {
            return $this->getMockData();
        }

        return Cache::remember('ga_realtime', 60, function () {
            try {
                $analytics = new AnalyticsData($this->client);
                
                $request = new \Google\Service\AnalyticsData\RunRealtimeReportRequest([
                    'dimensions' => [
                        new \Google\Service\AnalyticsData\Dimension(['name' => 'country']),
                        new \Google\Service\AnalyticsData\Dimension(['name' => 'deviceCategory']),
                    ],
                    'metrics' => [
                        new \Google\Service\AnalyticsData\Metric(['name' => 'activeUsers']),
                    ],
                ]);

                $response = $analytics->properties->runRealtimeReport("properties/{$this->propertyId}", $request);
                
                return $this->formatRealtimeResponse($response);
                
            } catch (\Exception $e) {
                Log::error('Google Analytics RealTime API Error: ' . $e->getMessage());
                return $this->getMockData();
            }
        });
    }

    public function getTrafficData($days = 30)
    {
        if (!$this->isConfigured()) {
            return $this->getMockTrafficData($days);
        }

        return Cache::remember("ga_traffic_{$days}", 3600, function () use ($days) {
            try {
                $analytics = new AnalyticsData($this->client);
                
                $request = new \Google\Service\AnalyticsData\RunReportRequest([
                    'dateRanges' => [
                        new \Google\Service\AnalyticsData\DateRange([
                            'startDate' => "{$days}DaysAgo",
                            'endDate' => 'today',
                        ]),
                    ],
                    'dimensions' => [
                        new \Google\Service\AnalyticsData\Dimension(['name' => 'date']),
                    ],
                    'metrics' => [
                        new \Google\Service\AnalyticsData\Metric(['name' => 'sessions']),
                        new \Google\Service\AnalyticsData\Metric(['name' => 'totalUsers']),
                        new \Google\Service\AnalyticsData\Metric(['name' => 'screenPageViews']),
                        new \Google\Service\AnalyticsData\Metric(['name' => 'bounceRate']),
                    ],
                ]);

                $response = $analytics->properties->runReport("properties/{$this->propertyId}", $request);
                
                return $this->formatTrafficResponse($response);
                
            } catch (\Exception $e) {
                Log::error('Google Analytics Traffic API Error: ' . $e->getMessage());
                return $this->getMockTrafficData($days);
            }
        });
    }

    public function getTopPages($days = 7)
    {
        if (!$this->isConfigured()) {
            return $this->getMockTopPages($days);
        }

        return Cache::remember("ga_toppages_{$days}", 3600, function () use ($days) {
            try {
                $analytics = new AnalyticsData($this->client);
                
                $request = new \Google\Service\AnalyticsData\RunReportRequest([
                    'dateRanges' => [
                        new \Google\Service\AnalyticsData\DateRange([
                            'startDate' => "{$days}DaysAgo",
                            'endDate' => 'today',
                        ]),
                    ],
                    'dimensions' => [
                        new \Google\Service\AnalyticsData\Dimension(['name' => 'pageTitle']),
                        new \Google\Service\AnalyticsData\Dimension(['name' => 'pagePath']),
                    ],
                    'metrics' => [
                        new \Google\Service\AnalyticsData\Metric(['name' => 'screenPageViews']),
                        new \Google\Service\AnalyticsData\Metric(['name' => 'averageSessionDuration']),
                    ],
                    'limit' => 10,
                ]);

                $response = $analytics->properties->runReport("properties/{$this->propertyId}", $request);
                
                return $this->formatPagesResponse($response);
                
            } catch (\Exception $e) {
                Log::error('Google Analytics Pages API Error: ' . $e->getMessage());
                return $this->getMockTopPages($days);
            }
        });
    }

    protected function formatRealtimeResponse($response)
    {
        $data = [
            'activeUsers' => 0,
            'countries' => [],
            'devices' => [],
        ];

        // Safely get totals
        $totals = $response->getTotals();
        if (!empty($totals) && isset($totals[0])) {
            $metricValues = $totals[0]->getMetricValues();
            if (!empty($metricValues) && isset($metricValues[0])) {
                $data['activeUsers'] = $metricValues[0]->getValue();
            }
        }

        // Safely process rows
        $rows = $response->getRows() ?? [];
        foreach ($rows as $row) {
            $dimensions = $row->getDimensionValues();
            $metrics = $row->getMetricValues();
            
            // Check if dimensions and metrics exist
            if (empty($dimensions) || empty($metrics)) {
                continue;
            }
            
            $country = isset($dimensions[0]) ? $dimensions[0]->getValue() : null;
            $device = isset($dimensions[1]) ? $dimensions[1]->getValue() : null;
            $users = isset($metrics[0]) ? $metrics[0]->getValue() : 0;

            if ($country && $country !== '(not set)') {
                $data['countries'][$country] = ($data['countries'][$country] ?? 0) + $users;
            }
            
            if ($device && $device !== '(not set)') {
                $data['devices'][$device] = ($data['devices'][$device] ?? 0) + $users;
            }
        }

        return $data;
    }

    protected function formatTrafficResponse($response)
    {
        $data = [];
        
        $rows = $response->getRows() ?? [];
        foreach ($rows as $row) {
            $dimensions = $row->getDimensionValues();
            $metrics = $row->getMetricValues();
            
            // Check if we have the expected data
            if (empty($dimensions) || empty($metrics) || count($metrics) < 4) {
                continue;
            }
            
            $data[] = [
                'date' => $dimensions[0]->getValue(),
                'sessions' => $metrics[0]->getValue(),
                'users' => $metrics[1]->getValue(),
                'pageviews' => $metrics[2]->getValue(),
                'bounce_rate' => $metrics[3]->getValue(),
            ];
        }

        return $data;
    }

    protected function formatPagesResponse($response)
    {
        $data = [];
        
        $rows = $response->getRows() ?? [];
        foreach ($rows as $row) {
            $dimensions = $row->getDimensionValues();
            $metrics = $row->getMetricValues();
            
            // Check if we have the expected data
            if (empty($dimensions) || empty($metrics) || count($dimensions) < 2 || count($metrics) < 2) {
                continue;
            }
            
            $data[] = [
                'title' => $dimensions[0]->getValue(),
                'path' => $dimensions[1]->getValue(),
                'views' => $metrics[0]->getValue(),
                'avg_duration' => $metrics[1]->getValue(),
            ];
        }

        return $data;
    }

    // Mock data methods for when API fails or is not configured
    protected function getMockData()
    {
        return [
            'activeUsers' => rand(5, 50),
            'countries' => [
                'United States' => rand(1, 10),
                'United Kingdom' => rand(1, 5),
                'Canada' => rand(1, 3)
            ],
            'devices' => [
                'desktop' => rand(1, 10),
                'mobile' => rand(1, 15),
                'tablet' => rand(1, 5)
            ]
        ];
    }

    protected function getMockTrafficData($days)
    {
        $data = [];
        for ($i = $days; $i >= 0; $i--) {
            $date = date('Y-m-d', strtotime("-$i days"));
            $data[] = [
                'date' => $date,
                'sessions' => rand(50, 200),
                'users' => rand(40, 180),
                'pageviews' => rand(100, 400),
                'bounce_rate' => round(rand(4000, 7000) / 100, 2),
            ];
        }
        return $data;
    }

    protected function getMockTopPages($days)
    {
        $pages = [
            ['Homepage', '/'],
            ['About Us', '/about'],
            ['Contact', '/contact'],
            ['Products', '/products'],
            ['Blog', '/blog'],
            ['FAQ', '/faq'],
            ['Services', '/services'],
            ['Pricing', '/pricing'],
            ['Documentation', '/docs'],
            ['Support', '/support']
        ];

        $data = [];
        foreach ($pages as $page) {
            $data[] = [
                'title' => $page[0],
                'path' => $page[1],
                'views' => rand(100, 1000),
                'avg_duration' => round(rand(30, 180), 1),
            ];
        }

        return $data;
    }

    // Test method to verify connection
    public function testConnection()
    {
        if (!$this->isConfigured()) {
            return ['success' => false, 'message' => 'Service not configured'];
        }

        try {
            $analytics = new AnalyticsData($this->client);
            
            // Simple test request
            $request = new \Google\Service\AnalyticsData\RunReportRequest([
                'dateRanges' => [
                    new \Google\Service\AnalyticsData\DateRange([
                        'startDate' => '7daysAgo',
                        'endDate' => 'today',
                    ]),
                ],
                'metrics' => [
                    new \Google\Service\AnalyticsData\Metric(['name' => 'sessions']),
                ],
            ]);

            $response = $analytics->properties->runReport("properties/{$this->propertyId}", $request);
            
            $rows = $response->getRows();
            $hasData = !empty($rows);
            
            return [
                'success' => true,
                'message' => $hasData ? 'Connected successfully with data' : 'Connected successfully but no data found',
                'has_data' => $hasData,
                'row_count' => $hasData ? count($rows) : 0
            ];
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    public function getTrafficChartData($days = 30)
    {
        if (!$this->isConfigured()) {
            return $this->getMockTrafficChartData($days);
        }

        return Cache::remember("ga_traffic_chart_{$days}", 3600, function () use ($days) {
            try {
                $analytics = new AnalyticsData($this->client);
                
                $request = new \Google\Service\AnalyticsData\RunReportRequest([
                    'dateRanges' => [
                        new \Google\Service\AnalyticsData\DateRange([
                            'startDate' => "{$days}DaysAgo",
                            'endDate' => 'today',
                        ]),
                    ],
                    'dimensions' => [
                        new \Google\Service\AnalyticsData\Dimension(['name' => 'date']),
                    ],
                    'metrics' => [
                        new \Google\Service\AnalyticsData\Metric(['name' => 'sessions']),
                        new \Google\Service\AnalyticsData\Metric(['name' => 'totalUsers']),
                    ],
                ]);

                $response = $analytics->properties->runReport("properties/{$this->propertyId}", $request);
                
                return $this->formatTrafficChartResponse($response);
                
            } catch (\Exception $e) {
                Log::error('Google Analytics Chart API Error: ' . $e->getMessage());
                return $this->getMockTrafficChartData($days);
            }
        });
    }

    public function getGeoData($days = 30)
    {
        if (!$this->isConfigured()) {
            return $this->getMockGeoData();
        }

        return Cache::remember("ga_geo_{$days}", 3600, function () use ($days) {
            try {
                $analytics = new AnalyticsData($this->client);
                
                $request = new \Google\Service\AnalyticsData\RunReportRequest([
                    'dateRanges' => [
                        new \Google\Service\AnalyticsData\DateRange([
                            'startDate' => "{$days}DaysAgo",
                            'endDate' => 'today',
                        ]),
                    ],
                    'dimensions' => [
                        new \Google\Service\AnalyticsData\Dimension(['name' => 'country']),
                        new \Google\Service\AnalyticsData\Dimension(['name' => 'city']),
                    ],
                    'metrics' => [
                        new \Google\Service\AnalyticsData\Metric(['name' => 'sessions']),
                        new \Google\Service\AnalyticsData\Metric(['name' => 'activeUsers']),
                    ],
                    'limit' => 50,
                ]);

                $response = $analytics->properties->runReport("properties/{$this->propertyId}", $request);
                
                return $this->formatGeoResponse($response);
                
            } catch (\Exception $e) {
                Log::error('Google Analytics Geo API Error: ' . $e->getMessage());
                return $this->getMockGeoData();
            }
        });
    }

    public function getDeviceData($days = 30)
    {
        if (!$this->isConfigured()) {
            return $this->getMockDeviceData();
        }

        return Cache::remember("ga_devices_{$days}", 3600, function () use ($days) {
            try {
                $analytics = new AnalyticsData($this->client);
                
                $request = new \Google\Service\AnalyticsData\RunReportRequest([
                    'dateRanges' => [
                        new \Google\Service\AnalyticsData\DateRange([
                            'startDate' => "{$days}DaysAgo",
                            'endDate' => 'today',
                        ]),
                    ],
                    'dimensions' => [
                        new \Google\Service\AnalyticsData\Dimension(['name' => 'deviceCategory']),
                    ],
                    'metrics' => [
                        new \Google\Service\AnalyticsData\Metric(['name' => 'sessions']),
                    ],
                ]);

                $response = $analytics->properties->runReport("properties/{$this->propertyId}", $request);
                
                return $this->formatDeviceResponse($response);
                
            } catch (\Exception $e) {
                Log::error('Google Analytics Device API Error: ' . $e->getMessage());
                return $this->getMockDeviceData();
            }
        });
    }

    protected function formatTrafficChartResponse($response)
    {
        $data = [
            'dates' => [],
            'sessions' => [],
            'users' => []
        ];

        $rows = $response->getRows() ?? [];
        foreach ($rows as $row) {
            $dimensions = $row->getDimensionValues();
            $metrics = $row->getMetricValues();
            
            if (empty($dimensions) || empty($metrics)) {
                continue;
            }
            
            $date = $dimensions[0]->getValue();
            $data['dates'][] = date('M j', strtotime($date));
            $data['sessions'][] = $metrics[0]->getValue();
            $data['users'][] = $metrics[1]->getValue();
        }

        return $data;
    }

    protected function formatGeoResponse($response)
    {
        $data = [];
        
        $rows = $response->getRows() ?? [];
        foreach ($rows as $row) {
            $dimensions = $row->getDimensionValues();
            $metrics = $row->getMetricValues();
            
            if (empty($dimensions) || empty($metrics)) {
                continue;
            }
            
            $country = $dimensions[0]->getValue();
            $city = $dimensions[1]->getValue();
            
            if ($country === '(not set)') continue;
            
            $data[] = [
                'country' => $country,
                'city' => $city !== '(not set)' ? $city : 'Unknown',
                'sessions' => $metrics[0]->getValue(),
                'users' => $metrics[1]->getValue(),
            ];
        }

        return $data;
    }

    protected function formatDeviceResponse($response)
    {
        $data = [];
        
        $rows = $response->getRows() ?? [];
        foreach ($rows as $row) {
            $dimensions = $row->getDimensionValues();
            $metrics = $row->getMetricValues();
            
            if (empty($dimensions) || empty($metrics)) {
                continue;
            }
            
            $device = $dimensions[0]->getValue();
            $sessions = $metrics[0]->getValue();
            
            if ($device !== '(not set)') {
                $data[] = [
                    'device' => ucfirst($device),
                    'sessions' => $sessions
                ];
            }
        }

        return $data;
    }

    // Mock data methods
    protected function getMockTrafficChartData($days)
    {
        $data = [
            'dates' => [],
            'sessions' => [],
            'users' => []
        ];

        for ($i = $days; $i >= 0; $i--) {
            $date = date('M j', strtotime("-$i days"));
            $data['dates'][] = $date;
            $data['sessions'][] = rand(50, 200);
            $data['users'][] = rand(40, 180);
        }

        return $data;
    }

    protected function getMockGeoData()
    {
        return [
            ['country' => 'United States', 'city' => 'New York', 'sessions' => rand(100, 500), 'users' => rand(80, 450)],
            ['country' => 'United Kingdom', 'city' => 'London', 'sessions' => rand(50, 200), 'users' => rand(40, 180)],
            ['country' => 'Canada', 'city' => 'Toronto', 'sessions' => rand(30, 150), 'users' => rand(25, 120)],
            ['country' => 'Australia', 'city' => 'Sydney', 'sessions' => rand(20, 100), 'users' => rand(15, 80)],
            ['country' => 'Germany', 'city' => 'Berlin', 'sessions' => rand(25, 120), 'users' => rand(20, 100)],
        ];
    }

    protected function getMockDeviceData()
    {
        return [
            ['device' => 'Desktop', 'sessions' => rand(200, 600)],
            ['device' => 'Mobile', 'sessions' => rand(300, 800)],
            ['device' => 'Tablet', 'sessions' => rand(50, 200)],
        ];
    }
}