<?php

namespace App\Services;

use Google\Client;
use Google\Service\SearchConsole;
use Illuminate\Support\Facades\Cache;

class GoogleSearchConsoleService
{
    protected $client;

    public function __construct()
    {
        $this->client = new Client();
        $this->setupClient();
    }

    protected function setupClient()
    {
        $this->client->setAuthConfig(env('GOOGLE_SERVICE_ACCOUNT_KEY_FILE'));
        $this->client->addScope(SearchConsole::WEBMASTERS_READONLY);
    }

    public function getSearchAnalytics($days = 30)
    {
        return Cache::remember("gsc_analytics_{$days}", 3600, function () use ($days) {
            $searchConsole = new SearchConsole($this->client);
            
            $request = new \Google\Service\SearchConsole\SearchAnalyticsQueryRequest([
                'startDate' => date('Y-m-d', strtotime("-$days days")),
                'endDate' => date('Y-m-d'),
                'dimensions' => ['date', 'query', 'page'],
                'rowLimit' => 1000,
            ]);

            $response = $searchConsole->searchanalytics->query(config('app.url'), $request);
            
            return $this->formatSearchResponse($response);
        });
    }

    public function getTopQueries($days = 30, $limit = 20)
    {
        return Cache::remember("gsc_queries_{$days}_{$limit}", 3600, function () use ($days, $limit) {
            $searchConsole = new SearchConsole($this->client);
            
            $request = new \Google\Service\SearchConsole\SearchAnalyticsQueryRequest([
                'startDate' => date('Y-m-d', strtotime("-$days days")),
                'endDate' => date('Y-m-d'),
                'dimensions' => ['query'],
                'rowLimit' => $limit,
            ]);

            $response = $searchConsole->searchanalytics->query(config('app.url'), $request);
            
            return $this->formatQueriesResponse($response);
        });
    }

    public function getTopPages($days = 30, $limit = 20)
    {
        return Cache::remember("gsc_pages_{$days}_{$limit}", 3600, function () use ($days, $limit) {
            $searchConsole = new SearchConsole($this->client);
            
            $request = new \Google\Service\SearchConsole\SearchAnalyticsQueryRequest([
                'startDate' => date('Y-m-d', strtotime("-$days days")),
                'endDate' => date('Y-m-d'),
                'dimensions' => ['page'],
                'rowLimit' => $limit,
            ]);

            $response = $searchConsole->searchanalytics->query(config('app.url'), $request);
            
            return $this->formatPagesResponse($response);
        });
    }

    protected function formatSearchResponse($response)
    {
        $data = [];
        
        if ($response->getRows()) {
            foreach ($response->getRows() as $row) {
                $data[] = [
                    'date' => $row->getKeys()[0],
                    'query' => $row->getKeys()[1] ?? 'N/A',
                    'page' => $row->getKeys()[2] ?? 'N/A',
                    'clicks' => $row->getClicks(),
                    'impressions' => $row->getImpressions(),
                    'ctr' => $row->getCtr(),
                    'position' => $row->getPosition(),
                ];
            }
        }

        return $data;
    }

    protected function formatQueriesResponse($response)
    {
        $data = [];
        
        if ($response->getRows()) {
            foreach ($response->getRows() as $row) {
                $data[] = [
                    'query' => $row->getKeys()[0],
                    'clicks' => $row->getClicks(),
                    'impressions' => $row->getImpressions(),
                    'ctr' => round($row->getCtr() * 100, 2),
                    'position' => round($row->getPosition(), 1),
                ];
            }
        }

        return $data;
    }

    protected function formatPagesResponse($response)
    {
        $data = [];
        
        if ($response->getRows()) {
            foreach ($response->getRows() as $row) {
                $data[] = [
                    'page' => $row->getKeys()[0],
                    'clicks' => $row->getClicks(),
                    'impressions' => $row->getImpressions(),
                    'ctr' => round($row->getCtr() * 100, 2),
                    'position' => round($row->getPosition(), 1),
                ];
            }
        }

        return $data;
    }
}